from django.views import generic
from django.contrib.auth.mixins import LoginRequiredMixin, PermissionRequiredMixin
from vaccine.models import Vaccine
from campaign.models import Campaign, Slot
from django.views import View
from vaccination.forms import VaccinationForm
from django.shortcuts import render
from django.http import HttpResponse, HttpResponseBadRequest, HttpResponseRedirect
from vaccination.models import Vaccination
from django.utils import timezone
from vaccination.utils import generate_pdf
from django.contrib.auth.decorators import login_required
from django.contrib import messages
from django.core.exceptions import PermissionDenied
from django.urls import reverse


class ChooseVaccine(LoginRequiredMixin, generic.ListView):
    model = Vaccine
    template_name = "vaccination/choose-vaccine.html"
    paginate_by = 10
    ordering = ["name"]


class ChooseCampaign(LoginRequiredMixin, generic.ListView):
    model = Campaign
    template_name = "vaccination/choose-campaign.html"
    paginate_by = 10
    ordering = ["start_date"]
    
    def get_queryset(self):
        return super().get_queryset().filter(vaccine=self.kwargs["vaccine_id"])


class ChooseSlot(LoginRequiredMixin, generic.ListView):
    model = Slot
    template_name = "vaccination/choose-slot.html"
    paginate_by = 10
    ordering = ["date"]
    
    def get_queryset(self):
        return super().get_queryset().filter(campaign=self.kwargs["campaign_id"], date__gte=timezone.now())


class ConfirmVaccination(View):
    form_class = VaccinationForm

    def get(self, request, *args, **kwargs):
        campaign = Campaign.objects.get(id=self.kwargs["campaign_id"])
        slot = Slot.objects.get(id=self.kwargs["slot_id"])
        form = self.form_class(initial={"patient": request.user, "campaign": campaign, "slot": slot})
        context = {
            "patient": request.user,
            "campaign": campaign,
            "slot": slot,
            "form": form
        }
        return render(request, "vaccination/confirm-vaccination.html", context)

    def post(self, request, *args, **kwargs):
        form = self.form_class(request.POST)
        if form.is_valid():
            is_reserved = Slot.reserve_vaccine(self.kwargs["campaign_id"], self.kwargs["slot_id"])
            if is_reserved:
                form.save()
                return HttpResponse("Congratulations! Your vaccination has been scheduled")
            return HttpResponse("Sorry! We are unable to reserve vaccine for you at this moment")
        return HttpResponseBadRequest("Invalid form data")


class VaccinationList(LoginRequiredMixin, generic.ListView):
    model = Vaccination
    template_name = "vaccination/vaccination-list-patient.html"
    paginate_by = 10
    ordering = ["id"]

    def get_queryset(self):
        return Vaccination.objects.filter(patient=self.request.user)


class VaccinationDetail(LoginRequiredMixin, generic.DetailView):
    model = Vaccination
    template_name = "vaccination/vaccination-detail.html"

@login_required
def appointment_letter(request, vaccination_id):
    vaccination = Vaccination.objects.get(id=vaccination_id)
    context = {
        "pdf_title": f"{vaccination.patient.get_full_name() } | Vaccination Appointment Letter",
        "date": str(timezone.now()),
        "title": "Appointment Letter",
        "subtitle": "To Whom It May Concern",
        "content": f"This is to inform that the {vaccination.campaign.vaccine.name } vaccination of Mr/Ms/Mrs {vaccination.patient.get_full_name() } is scheduled on { vaccination.slot.date }, { vaccination.slot.start_time } at { vaccination.campaign.center.name }.",
    }
    return generate_pdf(context)

@login_required
def vaccination_certificate(request, vaccination_id):
    vaccination = Vaccination.objects.get(id=vaccination_id)
    if vaccination.is_vaccinated:
        context = {
            "pdf_title": f"{vaccination.patient.get_full_name() } | Vaccine Certificate",
            "date": str(timezone.now()),
            "title": "Vaccine Certificate",
            "subtitle": "To Whom It May Concern",
            "content": f"This is to certify that Mr/Ms/Mrs {vaccination.patient.get_full_name() } has successfuly taken {vaccination.campaign.vaccine.name } vaccine on {vaccination.date}. The vaccination was scheduled on { vaccination.slot.date } { vaccination.slot.start_time } at { vaccination.campaign.center.name }.",
        }
        return generate_pdf(context)
    else:
        return HttpResponseBadRequest("User Not Vaccinated")


@login_required
def approve_vaccination(request, vaccination_id):
    """
    Approves the vaccination of patient
    """
    if request.user.has_perm("vaccination.change_vaccination"):
        try:
            vaccination = Vaccination.objects.get(id=vaccination_id)
        except Vaccination.DoesNotExist:
            return HttpResponseBadRequest("object not found with given id")
        
        if request.user in vaccination.campaign.agents.all():
            if vaccination.is_vaccinated:
                messages.info(request, "Vaccination is already Approved")
                return HttpResponseRedirect(reverse("vaccination:vaccination-detail", kwargs={"pk": vaccination_id}))
            else:
                vaccination.is_vaccinated = True
                vaccination.date = timezone.now().date()
                vaccination.updated_by = request.user
                vaccination.save()
                messages.success(request, "Vaccination approved successfully")
                return HttpResponseRedirect(reverse("vaccination:vaccination-detail", kwargs={"pk": vaccination_id}))
        else:
            messages.error(request, "You are not assigned to approve this vaccination")
            raise PermissionDenied()
    else:
        messages.error(request, "You don't have permission to approve vaccination")
        raise PermissionDenied()